#!/usr/bin/env python3
"""
Tusk.py Documentation Asset Generator
Generates all SVG assets and organizes README structure
"""

import os
import shutil
from pathlib import Path
from typing import Dict, List, Tuple
import json

class TuskAssetGenerator:
    """Generate and organize assets for Tusk.py documentation"""
    
    def __init__(self, base_path: str = "."):
        self.base_path = Path(base_path)
        self.assets_path = self.base_path / "assets"
        self.icons_path = self.assets_path / "icons"
        self.badges_path = self.assets_path / "badges"
        
        # Tusk.py color scheme
        self.colors = {
            'primary': '#FF6B6B',
            'secondary': '#4ECDC4',
            'accent': '#667eea',
            'success': '#2ecc71',
            'warning': '#f39c12',
            'danger': '#e74c3c',
            'info': '#3498db',
            'dark': '#2c3e50',
            'light': '#ecf0f1',
            'gray': '#95a5a6'
        }
    
    def setup_directories(self):
        """Create directory structure for assets"""
        directories = [
            self.assets_path,
            self.icons_path,
            self.badges_path,
            self.assets_path / "diagrams",
            self.assets_path / "screenshots",
            self.assets_path / "animations"
        ]
        
        for directory in directories:
            directory.mkdir(parents=True, exist_ok=True)
            print(f"✅ Created directory: {directory}")
    
    def create_main_logo(self):
        """Create the main Tusk.py logo"""
        logo_svg = f'''<svg width="400" height="100" viewBox="0 0 400 100" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="tuskGradient" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['primary']};stop-opacity:1" />
      <stop offset="100%" style="stop-color:{self.colors['secondary']};stop-opacity:1" />
    </linearGradient>
    <filter id="shadow">
      <feDropShadow dx="2" dy="2" stdDeviation="2" flood-opacity="0.3"/>
    </filter>
  </defs>
  
  <!-- Tusk Icon -->
  <g transform="translate(20, 20)">
    <path d="M30 10 L50 30 L30 50 L10 30 Z" fill="url(#tuskGradient)" filter="url(#shadow)"/>
    <path d="M30 20 L40 30 L30 40 L20 30 Z" fill="#fff" opacity="0.8"/>
    <circle cx="30" cy="30" r="3" fill="#fff"/>
  </g>
  
  <!-- Text -->
  <text x="80" y="55" font-family="Arial, sans-serif" font-size="48" font-weight="bold" fill="{self.colors['dark']}">
    Tusk.py
  </text>
  <text x="80" y="75" font-family="Arial, sans-serif" font-size="14" fill="{self.colors['gray']}">
    Revolutionary Python SDK for Tusk Integration
  </text>
</svg>'''
        
        self._save_svg('tusk-logo.svg', logo_svg)
    
    def create_performance_gauge(self):
        """Create animated performance gauge"""
        gauge_svg = f'''<svg width="400" height="300" viewBox="0 0 400 300" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="gaugeGradient" x1="0%" y1="0%" x2="100%" y2="0%">
      <stop offset="0%" style="stop-color:{self.colors['danger']}" />
      <stop offset="50%" style="stop-color:{self.colors['warning']}" />
      <stop offset="100%" style="stop-color:{self.colors['success']}" />
    </linearGradient>
  </defs>
  
  <!-- Title -->
  <text x="200" y="40" font-family="Arial" font-size="24" font-weight="bold" text-anchor="middle" fill="{self.colors['dark']}">
    Performance Boost
  </text>
  
  <!-- Gauge background -->
  <path d="M 50 250 A 150 150 0 0 1 350 250" fill="none" stroke="{self.colors['light']}" stroke-width="30"/>
  
  <!-- Gauge markers -->
  <g font-family="Arial" font-size="12" fill="{self.colors['gray']}">
    <text x="50" y="270" text-anchor="middle">0x</text>
    <text x="125" y="170" text-anchor="middle">15x</text>
    <text x="200" y="140" text-anchor="middle">30x</text>
    <text x="275" y="170" text-anchor="middle">45x</text>
    <text x="350" y="270" text-anchor="middle">60x</text>
  </g>
  
  <!-- Gauge fill (59x improvement = 98% of arc) -->
  <path d="M 50 250 A 150 150 0 0 1 340 220" fill="none" stroke="url(#gaugeGradient)" 
        stroke-width="30" stroke-linecap="round">
    <animate attributeName="stroke-dasharray" from="0 1000" to="450 1000" dur="2s" fill="freeze"/>
  </path>
  
  <!-- Center display -->
  <circle cx="200" cy="250" r="60" fill="{self.colors['dark']}"/>
  <text x="200" y="245" font-family="Arial" font-size="36" font-weight="bold" fill="#fff" text-anchor="middle">59x</text>
  <text x="200" y="265" font-family="Arial" font-size="12" fill="#fff" text-anchor="middle">FASTER</text>
</svg>'''
        
        self._save_svg('performance-gauge.svg', gauge_svg)
    
    def create_icons(self):
        """Create all icon SVGs"""
        icons = {
            'speed': f'''<path d="M13.5 2L2 7v10l11.5 5L21 19v-2.5l4 1.5v-11l-4 1.5V7l-7.5-5z" 
                          stroke="{self.colors['primary']}" stroke-width="2" fill="none" stroke-linecap="round"/>''',
            
            'rocket': f'''<path d="M12 2c0 0 0 7-3 10s-6 6-6 6 3-3 6-6 10-3 10-3-3 6-6 9-6 6-6 6 3-3 6-6c3-3 9-6 9-6" 
                          stroke="{self.colors['secondary']}" stroke-width="2" fill="none" stroke-linecap="round"/>
                          <circle cx="12" cy="12" r="3" fill="{self.colors['secondary']}"/>''',
            
            'ai': f'''<rect x="4" y="4" width="16" height="16" rx="2" stroke="{self.colors['accent']}" stroke-width="2" fill="none"/>
                      <circle cx="12" cy="12" r="3" stroke="{self.colors['accent']}" stroke-width="2" fill="none"/>
                      <path d="M12 1v6m0 6v6M1 12h6m6 0h6" stroke="{self.colors['accent']}" stroke-width="1"/>''',
            
            'database': f'''<ellipse cx="12" cy="5" rx="9" ry="3" stroke="{self.colors['info']}" stroke-width="2" fill="none"/>
                           <path d="M21 5v14c0 1.66-4.03 3-9 3s-9-1.34-9-3V5" stroke="{self.colors['info']}" stroke-width="2" fill="none"/>
                           <path d="M21 12c0 1.66-4.03 3-9 3s-9-1.34-9-3" stroke="{self.colors['info']}" stroke-width="2" fill="none"/>''',
            
            'shield': f'''<path d="M12 2l8 4v6c0 4-3.2 7.2-8 10-4.8-2.8-8-6-8-10V6l8-4z" 
                          stroke="{self.colors['success']}" stroke-width="2" fill="none" stroke-linejoin="round"/>
                          <path d="M9 12l2 2 4-4" stroke="{self.colors['success']}" stroke-width="2" 
                          fill="none" stroke-linecap="round" stroke-linejoin="round"/>''',
            
            'cloud': f'''<path d="M18 10h-1.26A8 8 0 109 20h9a5 5 0 000-10z" 
                         stroke="{self.colors['info']}" stroke-width="2" fill="none" stroke-linecap="round"/>''',
            
            'check': f'''<circle cx="12" cy="12" r="10" stroke="{self.colors['success']}" stroke-width="2" fill="none"/>
                        <path d="M8 12l2 2 4-4" stroke="{self.colors['success']}" stroke-width="2" 
                        fill="none" stroke-linecap="round" stroke-linejoin="round"/>''',
            
            'users': f'''<path d="M17 21v-2a4 4 0 00-4-4H5a4 4 0 00-4 4v2" stroke="{self.colors['primary']}" stroke-width="2" fill="none"/>
                        <circle cx="9" cy="7" r="4" stroke="{self.colors['primary']}" stroke-width="2" fill="none"/>
                        <path d="M23 21v-2a4 4 0 00-3-3.87" stroke="{self.colors['primary']}" stroke-width="2" fill="none"/>
                        <path d="M16 3.13a4 4 0 010 7.75" stroke="{self.colors['primary']}" stroke-width="2" fill="none"/>''',
            
            'lock': f'''<rect x="3" y="11" width="18" height="11" rx="2" stroke="{self.colors['warning']}" stroke-width="2" fill="none"/>
                       <path d="M7 11V7a5 5 0 0110 0v4" stroke="{self.colors['warning']}" stroke-width="2" fill="none"/>''',
            
            'api': f'''<path d="M4 4h6v6H4zM14 4h6v6h-6zM14 14h6v6h-6zM4 14h6v6H4z" 
                      stroke="{self.colors['accent']}" stroke-width="2" fill="none"/>
                      <path d="M10 7h4M10 17h4M7 10v4M17 10v4" stroke="{self.colors['accent']}" stroke-width="2"/>'''
        }
        
        for name, path_data in icons.items():
            icon_svg = f'''<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
  {path_data}
</svg>'''
            self._save_svg(f'icons/{name}.svg', icon_svg)
    
    def create_feature_cards(self):
        """Create feature card illustrations"""
        features = [
            ('performance', self.colors['danger'], '⚡', 'Lightning Fast'),
            ('ai-ml', self.colors['accent'], '🤖', 'AI Powered'),
            ('enterprise', self.colors['success'], '🏢', 'Enterprise Ready'),
            ('operators', self.colors['info'], '🔧', '85+ Operators')
        ]
        
        for name, color, emoji, title in features:
            card_svg = f'''<svg width="300" height="200" viewBox="0 0 300 200" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="cardGrad{name}" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:{color};stop-opacity:0.8" />
      <stop offset="100%" style="stop-color:{color};stop-opacity:0.4" />
    </linearGradient>
    <filter id="cardShadow">
      <feDropShadow dx="0" dy="4" stdDeviation="6" flood-opacity="0.15"/>
    </filter>
  </defs>
  
  <rect x="10" y="10" width="280" height="180" rx="15" fill="url(#cardGrad{name})" filter="url(#cardShadow)"/>
  <rect x="20" y="20" width="260" height="160" rx="10" fill="#fff" opacity="0.95"/>
  
  <text x="150" y="80" font-family="Arial" font-size="48" text-anchor="middle">{emoji}</text>
  <text x="150" y="130" font-family="Arial" font-size="24" font-weight="bold" fill="{color}" text-anchor="middle">{title}</text>
</svg>'''
            self._save_svg(f'{name}-card.svg', card_svg)
    
    def create_workflow_diagram(self):
        """Create installation workflow diagram"""
        workflow_svg = f'''<svg width="900" height="300" viewBox="0 0 900 300" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <filter id="workflowShadow">
      <feDropShadow dx="2" dy="2" stdDeviation="3" flood-opacity="0.2"/>
    </filter>
  </defs>
  
  <!-- Title -->
  <text x="450" y="40" font-family="Arial" font-size="28" font-weight="bold" 
        text-anchor="middle" fill="{self.colors['dark']}">Quick Start in 4 Steps</text>
  
  <!-- Step boxes -->
  <g id="steps">
    <!-- Step 1 -->
    <g transform="translate(50, 80)">
      <rect width="180" height="140" rx="10" fill="{self.colors['info']}" filter="url(#workflowShadow)"/>
      <text x="90" y="40" font-family="Arial" font-size="48" font-weight="bold" fill="#fff" text-anchor="middle">1</text>
      <text x="90" y="80" font-family="Arial" font-size="18" fill="#fff" text-anchor="middle">Install</text>
      <text x="90" y="105" font-family="monospace" font-size="12" fill="#fff" text-anchor="middle">pip install</text>
      <text x="90" y="120" font-family="monospace" font-size="12" fill="#fff" text-anchor="middle">tusktsk[full]</text>
    </g>
    
    <!-- Arrow 1 -->
    <path d="M 240 150 L 290 150" stroke="{self.colors['gray']}" stroke-width="3" 
          marker-end="url(#arrowhead)" stroke-dasharray="5,5">
      <animate attributeName="stroke-dashoffset" from="10" to="0" dur="1s" repeatCount="indefinite"/>
    </path>
    
    <!-- Step 2 -->
    <g transform="translate(300, 80)">
      <rect width="180" height="140" rx="10" fill="{self.colors['success']}" filter="url(#workflowShadow)"/>
      <text x="90" y="40" font-family="Arial" font-size="48" font-weight="bold" fill="#fff" text-anchor="middle">2</text>
      <text x="90" y="80" font-family="Arial" font-size="18" fill="#fff" text-anchor="middle">Import</text>
      <text x="90" y="105" font-family="monospace" font-size="12" fill="#fff" text-anchor="middle">from tusk import</text>
      <text x="90" y="120" font-family="monospace" font-size="12" fill="#fff" text-anchor="middle">TuskFlask</text>
    </g>
    
    <!-- Arrow 2 -->
    <path d="M 490 150 L 540 150" stroke="{self.colors['gray']}" stroke-width="3" 
          marker-end="url(#arrowhead)" stroke-dasharray="5,5">
      <animate attributeName="stroke-dashoffset" from="10" to="0" dur="1s" repeatCount="indefinite"/>
    </path>
    
    <!-- Step 3 -->
    <g transform="translate(550, 80)">
      <rect width="180" height="140" rx="10" fill="{self.colors['warning']}" filter="url(#workflowShadow)"/>
      <text x="90" y="40" font-family="Arial" font-size="48" font-weight="bold" fill="#fff" text-anchor="middle">3</text>
      <text x="90" y="80" font-family="Arial" font-size="18" fill="#fff" text-anchor="middle">Initialize</text>
      <text x="90" y="105" font-family="monospace" font-size="12" fill="#fff" text-anchor="middle">tusk =</text>
      <text x="90" y="120" font-family="monospace" font-size="12" fill="#fff" text-anchor="middle">TuskFlask(app)</text>
    </g>
    
    <!-- Arrow 3 -->
    <path d="M 740 150 L 790 150" stroke="{self.colors['gray']}" stroke-width="3" 
          marker-end="url(#arrowhead)" stroke-dasharray="5,5">
      <animate attributeName="stroke-dashoffset" from="10" to="0" dur="1s" repeatCount="indefinite"/>
    </path>
    
    <!-- Step 4 -->
    <g transform="translate(800, 120)">
      <circle cx="40" cy="40" r="35" fill="{self.colors['danger']}" filter="url(#workflowShadow)"/>
      <text x="40" y="50" font-family="Arial" font-size="24" font-weight="bold" fill="#fff" text-anchor="middle">🚀</text>
    </g>
  </g>
  
  <!-- Result text -->
  <text x="450" y="270" font-family="Arial" font-size="20" font-weight="bold" 
        fill="{self.colors['success']}" text-anchor="middle">59x Faster Performance!</text>
  
  <defs>
    <marker id="arrowhead" markerWidth="10" markerHeight="10" refX="9" refY="3" orient="auto">
      <polygon points="0 0, 10 3, 0 6" fill="{self.colors['gray']}"/>
    </marker>
  </defs>
</svg>'''
        
        self._save_svg('quickstart-workflow.svg', workflow_svg)
    
    def create_tech_stack_diagram(self):
        """Create technology stack visualization"""
        stack_svg = f'''<svg width="800" height="600" viewBox="0 0 800 600" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <pattern id="grid" width="20" height="20" patternUnits="userSpaceOnUse">
      <circle cx="10" cy="10" r="1" fill="{self.colors['light']}"/>
    </pattern>
    <filter id="glow">
      <feGaussianBlur stdDeviation="3" result="coloredBlur"/>
      <feMerge>
        <feMergeNode in="coloredBlur"/>
        <feMergeNode in="SourceGraphic"/>
      </feMerge>
    </filter>
  </defs>
  
  <!-- Background -->
  <rect width="800" height="600" fill="#f8f9fa"/>
  <rect width="800" height="600" fill="url(#grid)"/>
  
  <!-- Core -->
  <g transform="translate(400, 300)">
    <circle r="100" fill="{self.colors['danger']}" filter="url(#glow)"/>
    <text y="-10" font-family="Arial" font-size="28" font-weight="bold" fill="#fff" text-anchor="middle">Tusk.py</text>
    <text y="20" font-family="Arial" font-size="16" fill="#fff" text-anchor="middle">Core Engine</text>
  </g>
  
  <!-- Orbiting components -->
  <g transform="translate(400, 300)">
    <!-- Performance Layer -->
    <g transform="rotate(0)">
      <g transform="translate(180, 0)">
        <circle r="50" fill="{self.colors['warning']}" opacity="0.9"/>
        <text font-family="Arial" font-size="14" fill="#fff" text-anchor="middle">Performance</text>
      </g>
      <animateTransform attributeName="transform" type="rotate" from="0 0 0" to="360 0 0" dur="30s" repeatCount="indefinite"/>
    </g>
    
    <!-- AI/ML Layer -->
    <g transform="rotate(60)">
      <g transform="translate(180, 0)">
        <circle r="50" fill="{self.colors['accent']}" opacity="0.9"/>
        <text font-family="Arial" font-size="14" fill="#fff" text-anchor="middle">AI/ML</text>
      </g>
      <animateTransform attributeName="transform" type="rotate" from="60 0 0" to="420 0 0" dur="30s" repeatCount="indefinite"/>
    </g>
    
    <!-- Enterprise Layer -->
    <g transform="rotate(120)">
      <g transform="translate(180, 0)">
        <circle r="50" fill="{self.colors['success']}" opacity="0.9"/>
        <text font-family="Arial" font-size="14" fill="#fff" text-anchor="middle">Enterprise</text>
      </g>
      <animateTransform attributeName="transform" type="rotate" from="120 0 0" to="480 0 0" dur="30s" repeatCount="indefinite"/>
    </g>
    
    <!-- Database Layer -->
    <g transform="rotate(180)">
      <g transform="translate(180, 0)">
        <circle r="50" fill="{self.colors['info']}" opacity="0.9"/>
        <text font-family="Arial" font-size="14" fill="#fff" text-anchor="middle">Database</text>
      </g>
      <animateTransform attributeName="transform" type="rotate" from="180 0 0" to="540 0 0" dur="30s" repeatCount="indefinite"/>
    </g>
    
    <!-- Cloud Layer -->
    <g transform="rotate(240)">
      <g transform="translate(180, 0)">
        <circle r="50" fill="{self.colors['secondary']}" opacity="0.9"/>
        <text font-family="Arial" font-size="14" fill="#fff" text-anchor="middle">Cloud</text>
      </g>
      <animateTransform attributeName="transform" type="rotate" from="240 0 0" to="600 0 0" dur="30s" repeatCount="indefinite"/>
    </g>
    
    <!-- Security Layer -->
    <g transform="rotate(300)">
      <g transform="translate(180, 0)">
        <circle r="50" fill="{self.colors['dark']}" opacity="0.9"/>
        <text font-family="Arial" font-size="14" fill="#fff" text-anchor="middle">Security</text>
      </g>
      <animateTransform attributeName="transform" type="rotate" from="300 0 0" to="660 0 0" dur="30s" repeatCount="indefinite"/>
    </g>
  </g>
  
  <!-- Title -->
  <text x="400" y="50" font-family="Arial" font-size="32" font-weight="bold" 
        text-anchor="middle" fill="{self.colors['dark']}">Complete Technology Stack</text>
  
  <!-- Labels -->
  <g font-family="Arial" font-size="14" fill="{self.colors['gray']}">
    <text x="50" y="580">Flask Integration</text>
    <text x="650" y="580">85+ Operators</text>
  </g>
</svg>'''
        
        self._save_svg('tech-stack.svg', stack_svg)
    
    def create_badges(self):
        """Create badge SVGs"""
        badges = [
            ('python', '3.8+', self.colors['info']),
            ('version', '1.0.0', self.colors['success']),
            ('tests', 'passing', self.colors['success']),
            ('coverage', '95%', self.colors['success']),
            ('performance', '59x', self.colors['danger']),
            ('license', 'MIT', self.colors['success']),
            ('downloads', '10k/month', self.colors['info']),
            ('stars', '1.2k', self.colors['warning'])
        ]
        
        for name, value, color in badges:
            badge_svg = self._create_badge(name.upper(), value, color)
            self._save_svg(f'badges/{name}.svg', badge_svg)
    
    def _create_badge(self, label: str, value: str, color: str) -> str:
        """Create a badge SVG"""
        label_width = len(label) * 8 + 20
        value_width = len(value) * 8 + 20
        total_width = label_width + value_width
        
        return f'''<svg width="{total_width}" height="20" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="bg" x2="0" y2="100%">
      <stop offset="0" stop-color="#bbb" stop-opacity=".1"/>
      <stop offset="1" stop-opacity=".1"/>
    </linearGradient>
  </defs>
  <rect width="{total_width}" height="20" rx="3" fill="#555"/>
  <rect x="{label_width}" width="{value_width}" height="20" rx="3" fill="{color}"/>
  <rect width="{total_width}" height="20" rx="3" fill="url(#bg)"/>
  <text x="{label_width/2}" y="14" font-family="Arial" font-size="11" fill="#fff" text-anchor="middle">{label}</text>
  <text x="{label_width + value_width/2}" y="14" font-family="Arial" font-size="11" fill="#fff" text-anchor="middle">{value}</text>
</svg>'''
    
    def _save_svg(self, filename: str, content: str):
        """Save SVG content to file"""
        filepath = self.assets_path / filename
        filepath.parent.mkdir(parents=True, exist_ok=True)
        filepath.write_text(content)
        print(f"✅ Created: {filepath}")
    
    def create_readme_structure(self):
        """Create organized README structure"""
        readme_template = '''# Tusk.py - README Structure

## 📁 Recommended Project Structure

```
tusk-py/
├── README.md                 # Main documentation
├── CONTRIBUTING.md          # Contribution guidelines
├── LICENSE                  # MIT License
├── CHANGELOG.md            # Version history
├── setup.py                # Package setup
├── requirements.txt        # Dependencies
├── .github/               
│   ├── workflows/         # GitHub Actions
│   ├── ISSUE_TEMPLATE/    # Issue templates
│   └── PULL_REQUEST_TEMPLATE.md
├── assets/                # Documentation assets
│   ├── tusk-logo.svg
│   ├── hero-banner.svg
│   ├── icons/            # Icon library
│   ├── badges/           # Status badges
│   └── diagrams/         # Architecture diagrams
├── docs/                 # Detailed documentation
│   ├── quickstart.md
│   ├── api-reference.md
│   ├── examples.md
│   └── enterprise.md
├── examples/             # Example implementations
│   ├── basic/
│   ├── advanced/
│   └── enterprise/
├── tests/               # Test suite
│   ├── test_performance.py
│   ├── test_ai_components.py
│   └── test_integration.py
└── tusktsk/            # Main package
    ├── __init__.py
    ├── core.py
    ├── ai/
    ├── enterprise/
    └── operators/
```

## 📝 README.md Sections Order

1. **Header**
   - Logo
   - Badges
   - Tagline
   - Quick links

2. **Hero Section**
   - Key metrics
   - Visual impact
   - Value proposition

3. **Table of Contents**
   - Auto-generated
   - Collapsible

4. **Overview**
   - What is Tusk.py?
   - Key features grid
   - Why choose Tusk.py?

5. **Quick Start**
   - Installation options
   - Basic example
   - Immediate value

6. **Features**
   - Performance
   - AI/ML
   - Enterprise
   - Operators

7. **Documentation**
   - Links hub
   - API reference
   - Examples

8. **Community**
   - Contributing
   - Support
   - Roadmap

9. **Footer**
   - License
   - Credits
   - Links
'''
        
        (self.base_path / 'README_STRUCTURE.md').write_text(readme_template)
        print("✅ Created README structure guide")
    
    def generate_all(self):
        """Generate all assets"""
        print("🚀 Starting Tusk.py asset generation...")
        
        # Setup
        self.setup_directories()
        
        # Create assets
        self.create_main_logo()
        self.create_performance_gauge()
        self.create_icons()
        self.create_feature_cards()
        self.create_workflow_diagram()
        self.create_tech_stack_diagram()
        self.create_badges()
        self.create_readme_structure()
        
        # Summary
        print("\n✨ Asset generation complete!")
        print(f"📁 Assets created in: {self.assets_path}")
        print("\nNext steps:")
        print("1. Review generated assets")
        print("2. Customize colors/text as needed")
        print("3. Commit to your repository")
        print("4. Update README.md with asset links")


def main():
    """Main entry point"""
    import argparse
    
    parser = argparse.ArgumentParser(description='Generate Tusk.py documentation assets')
    parser.add_argument('--path', default='.', help='Base path for asset generation')
    parser.add_argument('--custom-colors', type=str, help='JSON file with custom colors')
    
    args = parser.parse_args()
    
    generator = TuskAssetGenerator(args.path)
    
    # Load custom colors if provided
    if args.custom_colors:
        with open(args.custom_colors, 'r') as f:
            custom_colors = json.load(f)
            generator.colors.update(custom_colors)
    
    generator.generate_all()


if __name__ == '__main__':
    main()
